import { useEffect, useRef } from 'react';
import { ArrowRight } from 'lucide-react';
import gsap from 'gsap';
import { ScrollTrigger } from 'gsap/ScrollTrigger';

gsap.registerPlugin(ScrollTrigger);

const projects = [
  {
    number: '01',
    title: 'OPTIMISATION DE LA GESTION DE MAGASIN',
    problem: 'Un opérateur local, confronté à une gestion fragmentée de son inventaire et de ses ventes, souhaitait unifier ses opérations.',
    solution: 'Nous avons intégré notre solution de gestion de point de vente (POS), formé son personnel et mis en place un support réactif.',
    result: 'Amélioration du suivi des flux, réduction significative des erreurs de caisse, visibilité en temps réel sur la performance des points de vente et reporting clair.',
  },
  {
    number: '02',
    title: 'DÉMATÉRIALISATION STRATÉGIQUE',
    problem: 'Une institution souhaite moderniser ses processus internes en adoptant un système de gestion documentaire sans papier.',
    solution: 'Nous avons mis en place une GED modulaire couvrant la gestion de la correspondance, la circulation des documents administratifs et l\'archivage réglementaire. Formation, migration du contenu critique et accompagnement au changement.',
    result: 'L\'institution a réduit les délais de traitement, sécurisé ses flux internes et renforcé la traçabilité documentaire entre ses services.',
  },
  {
    number: '03',
    title: 'STRUCTURATION DES SERVICES IT',
    problem: 'L\'organisation faisait face à une prolifération des canaux de support (emails, appels, demandes informelles), limitant la visibilité, la traçabilité et le respect des accords de niveau de service (SLA).',
    solution: 'Nous avons déployé une plateforme ITSM on-premise, intégrant un module Helpdesk (incidents, demandes, SLA), la gestion des actifs IT, des workflows métier personnalisés et des processus alignés sur les bonnes pratiques ITIL.',
    result: 'Une plateforme unique pour toutes les demandes IT, réduction des délais de traitement, qualité de service améliorée et gouvernance IT claire, soutenant une adoption durable par les utilisateurs.',
  },
];

export default function Projects() {
  const sectionRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const ctx = gsap.context(() => {
      gsap.fromTo(
        '.project-card',
        { y: 50, opacity: 0 },
        {
          y: 0,
          opacity: 1,
          duration: 0.6,
          stagger: 0.15,
          scrollTrigger: {
            trigger: '.projects-grid',
            start: 'top 85%',
            toggleActions: 'play none none reverse',
          },
        }
      );
    }, sectionRef);

    return () => ctx.revert();
  }, []);

  return (
    <section ref={sectionRef} id="projects" className="section-padding bg-[#0a0a0a] relative">
      <div className="container-custom">
        {/* Section Header */}
        <div className="mb-12">
          <span className="text-sm text-white/40 uppercase tracking-widest mb-4 block">
            Projets
          </span>
          <h2 className="text-3xl md:text-4xl lg:text-5xl font-bold text-white mb-6">
            NOS <span className="gradient-text">PROJETS</span>
          </h2>
          <p className="text-white/60 text-lg max-w-3xl">
            Chez ZLO Technologies, chaque projet est une opportunité de transformer les défis opérationnels en solutions concrètes, durables et adaptées au terrain avec un focus sur la valeur métier, l'adoption par les utilisateurs et la performance opérationnelle.
          </p>
          <p className="text-white/60 mt-4">
            Voici quelques missions récentes menées avec nos clients à travers différents secteurs.
          </p>
        </div>

        {/* Projects Grid */}
        <div className="projects-grid space-y-8">
          {projects.map((project, index) => (
            <div
              key={index}
              className="project-card card-hover p-8 lg:p-10"
            >
              <div className="flex flex-col lg:flex-row lg:items-start gap-6 lg:gap-10">
                {/* Number */}
                <div className="text-4xl lg:text-5xl font-bold text-white/10">
                  {project.number}
                </div>

                {/* Content */}
                <div className="flex-1">
                  <h3 className="text-xl lg:text-2xl font-bold text-white mb-6">
                    {project.title}
                  </h3>

                  <div className="grid md:grid-cols-3 gap-6">
                    <div>
                      <span className="text-red-500 text-sm font-medium uppercase tracking-wider mb-2 block">
                        Problem
                      </span>
                      <p className="text-white/60 text-sm leading-relaxed">
                        {project.problem}
                      </p>
                    </div>

                    <div>
                      <span className="text-orange-500 text-sm font-medium uppercase tracking-wider mb-2 block">
                        Solution
                      </span>
                      <p className="text-white/60 text-sm leading-relaxed">
                        {project.solution}
                      </p>
                    </div>

                    <div>
                      <span className="text-green-500 text-sm font-medium uppercase tracking-wider mb-2 block">
                        Result
                      </span>
                      <p className="text-white/60 text-sm leading-relaxed">
                        {project.result}
                      </p>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          ))}
        </div>

        {/* CTA */}
        <div className="mt-16 text-center">
          <p className="text-white/40 text-lg mb-6">
            Entreprises, collectivités et institutions : transformons vos processus ensemble.
          </p>
          <button className="btn-primary">
            <span>Prendre rendez-vous</span>
            <ArrowRight className="w-5 h-5" />
          </button>
        </div>
      </div>
    </section>
  );
}
