<?php
/**
 * ZLO Platform - Setting Model
 */

declare(strict_types=1);

require_once __DIR__ . '/../core/Model.php';

class Setting
{
    private Database $db;
    private ?array $cache = null;
    
    public function __construct()
    {
        $this->db = new Database();
    }
    
    /**
     * Get all settings
     */
    public function getAll(): array
    {
        if ($this->cache === null) {
            $this->cache = $this->db->fetchOne("SELECT * FROM settings LIMIT 1") ?: [];
        }
        
        return $this->cache;
    }
    
    /**
     * Get single setting
     */
    public function get(string $key, $default = null)
    {
        $settings = $this->getAll();
        return $settings[$key] ?? $default;
    }
    
    /**
     * Update settings
     */
    public function update(array $data): bool
    {
        // Filter allowed fields
        $allowedFields = [
            'site_name', 'site_tagline', 'logo', 'logo_white', 'favicon',
            'contact_email', 'contact_phone', 'contact_address',
            'social_facebook', 'social_twitter', 'social_linkedin', 
            'social_instagram', 'social_youtube',
            'google_analytics_id', 'google_tag_manager', 'facebook_pixel',
            'meta_default_title', 'meta_default_description', 'meta_default_keywords',
            'custom_css', 'custom_js', 'maintenance_mode', 'maintenance_message'
        ];
        
        $updateData = array_intersect_key($data, array_flip($allowedFields));
        
        if (empty($updateData)) {
            return false;
        }
        
        // Build SET clause
        $setClause = [];
        $params = [];
        
        foreach ($updateData as $key => $value) {
            $setClause[] = "{$key} = ?";
            $params[] = $value;
        }
        
        $sql = "UPDATE settings SET " . implode(', ', $setClause) . ", updated_at = NOW() WHERE id = 1";
        
        $result = $this->db->query($sql, $params);
        
        // Clear cache
        $this->cache = null;
        
        return $result;
    }
    
    /**
     * Get social links
     */
    public function getSocialLinks(): array
    {
        $settings = $this->getAll();
        
        return [
            'facebook' => $settings['social_facebook'] ?? null,
            'twitter' => $settings['social_twitter'] ?? null,
            'linkedin' => $settings['social_linkedin'] ?? null,
            'instagram' => $settings['social_instagram'] ?? null,
            'youtube' => $settings['social_youtube'] ?? null
        ];
    }
    
    /**
     * Get contact info
     */
    public function getContactInfo(): array
    {
        $settings = $this->getAll();
        
        return [
            'email' => $settings['contact_email'] ?? 'contact@zlotech.net',
            'phone' => $settings['contact_phone'] ?? '+228 92 60 59 49',
            'address' => $settings['contact_address'] ?? 'Lomé, Togo'
        ];
    }
    
    /**
     * Get SEO defaults
     */
    public function getSeoDefaults(): array
    {
        $settings = $this->getAll();
        
        return [
            'title' => $settings['meta_default_title'] ?? $settings['site_name'] ?? 'ZLO Technologies',
            'description' => $settings['meta_default_description'] ?? '',
            'keywords' => $settings['meta_default_keywords'] ?? ''
        ];
    }
    
    /**
     * Get tracking codes
     */
    public function getTrackingCodes(): array
    {
        $settings = $this->getAll();
        
        return [
            'google_analytics' => $settings['google_analytics_id'] ?? null,
            'google_tag_manager' => $settings['google_tag_manager'] ?? null,
            'facebook_pixel' => $settings['facebook_pixel'] ?? null
        ];
    }
    
    /**
     * Check maintenance mode
     */
    public function isMaintenanceMode(): bool
    {
        return (bool) $this->get('maintenance_mode', false);
    }
    
    /**
     * Get maintenance message
     */
    public function getMaintenanceMessage(): string
    {
        return $this->get('maintenance_message', 'Site is under maintenance. Please check back later.');
    }
    
    /**
     * Toggle maintenance mode
     */
    public function toggleMaintenance(bool $enabled): bool
    {
        return $this->update(['maintenance_mode' => $enabled ? 1 : 0]);
    }
    
    /**
     * Get site info
     */
    public function getSiteInfo(): array
    {
        $settings = $this->getAll();
        
        return [
            'name' => $settings['site_name'] ?? 'ZLO Technologies',
            'tagline' => $settings['site_tagline'] ?? 'Digital Innovation for Africa',
            'logo' => $settings['logo'] ?? null,
            'logo_white' => $settings['logo_white'] ?? null,
            'favicon' => $settings['favicon'] ?? null
        ];
    }
    
    /**
     * Get custom code
     */
    public function getCustomCode(): array
    {
        $settings = $this->getAll();
        
        return [
            'css' => $settings['custom_css'] ?? '',
            'js' => $settings['custom_js'] ?? ''
        ];
    }
}
