<?php
/**
 * ZLO Platform - Setting Controller
 */

declare(strict_types=1);

require_once __DIR__ . '/../models/Setting.php';
require_once __DIR__ . '/../core/Router.php';

class SettingController
{
    private Setting $settingModel;
    
    public function __construct()
    {
        $this->settingModel = new Setting();
    }
    
    /**
     * Get all settings (admin)
     */
    public function index(array $params): array
    {
        AuthController::requirePermission('settings.view');
        
        return $this->settingModel->getAll();
    }
    
    /**
     * Update settings (admin)
     */
    public function update(array $params): array
    {
        AuthController::requirePermission('settings.edit');
        
        $body = Router::getBody();
        $security = new Security();
        
        // Sanitize inputs
        $data = [];
        
        $stringFields = [
            'site_name', 'site_tagline', 'logo', 'logo_white', 'favicon',
            'contact_email', 'contact_phone', 'contact_address',
            'social_facebook', 'social_twitter', 'social_linkedin', 
            'social_instagram', 'social_youtube',
            'google_analytics_id', 'google_tag_manager', 'facebook_pixel',
            'meta_default_title', 'meta_default_description', 'meta_default_keywords',
            'maintenance_message'
        ];
        
        foreach ($stringFields as $field) {
            if (isset($body[$field])) {
                $data[$field] = $security->sanitize($body[$field]);
            }
        }
        
        // Code fields (don't sanitize)
        if (isset($body['custom_css'])) {
            $data['custom_css'] = $body['custom_css'];
        }
        if (isset($body['custom_js'])) {
            $data['custom_js'] = $body['custom_js'];
        }
        
        // Boolean fields
        if (isset($body['maintenance_mode'])) {
            $data['maintenance_mode'] = $body['maintenance_mode'] ? 1 : 0;
        }
        
        try {
            $this->settingModel->update($data);
            return $this->settingModel->getAll();
        } catch (Exception $e) {
            http_response_code(500);
            return ['error' => 'Failed to update settings'];
        }
    }
    
    /**
     * Get public settings
     */
    public function public(array $params): array
    {
        return [
            'site' => $this->settingModel->getSiteInfo(),
            'contact' => $this->settingModel->getContactInfo(),
            'social' => $this->settingModel->getSocialLinks()
        ];
    }
    
    /**
     * Get SEO settings
     */
    public function seo(array $params): array
    {
        return $this->settingModel->getSeoDefaults();
    }
    
    /**
     * Get tracking codes (admin only)
     */
    public function tracking(array $params): array
    {
        AuthController::requirePermission('settings.view');
        
        return $this->settingModel->getTrackingCodes();
    }
    
    /**
     * Toggle maintenance mode
     */
    public function toggleMaintenance(array $params): array
    {
        AuthController::requirePermission('settings.edit');
        
        $body = Router::getBody();
        $enabled = $body['enabled'] ?? false;
        
        try {
            $this->settingModel->toggleMaintenance($enabled);
            return [
                'message' => $enabled ? 'Maintenance mode enabled' : 'Maintenance mode disabled',
                'maintenance_mode' => $enabled
            ];
        } catch (Exception $e) {
            http_response_code(500);
            return ['error' => 'Failed to toggle maintenance mode'];
        }
    }
    
    /**
     * Get contact info (public)
     */
    public function contact(array $params): array
    {
        return $this->settingModel->getContactInfo();
    }
    
    /**
     * Get social links (public)
     */
    public function social(array $params): array
    {
        $social = $this->settingModel->getSocialLinks();
        
        // Filter out null values
        return array_filter($social, function($value) {
            return $value !== null;
        });
    }
}
