import { useEffect, useState, useCallback } from 'react';
import { toast } from 'sonner';

// Version actuelle du site
const SITE_VERSION = '1.0.0';

interface ServiceWorkerState {
  isUpdateAvailable: boolean;
  isOfflineReady: boolean;
  updateApp: () => void;
}

export function useServiceWorker(): ServiceWorkerState {
  const [isUpdateAvailable, setIsUpdateAvailable] = useState(false);
  const [isOfflineReady, setIsOfflineReady] = useState(false);
  const [waitingWorker, setWaitingWorker] = useState<ServiceWorker | null>(null);

  const updateApp = useCallback(() => {
    if (waitingWorker) {
      waitingWorker.postMessage({ type: 'SKIP_WAITING' });
      window.location.reload();
    }
  }, [waitingWorker]);

  useEffect(() => {
    // Vérifier la version au chargement
    const checkVersion = () => {
      const storedVersion = localStorage.getItem('zlo_site_version');
      if (storedVersion && storedVersion !== SITE_VERSION) {
        console.log(`[Version] Update detected: ${storedVersion} -> ${SITE_VERSION}`);
        // Forcer le rechargement du cache
        if ('caches' in window) {
          caches.keys().then((cacheNames) => {
            cacheNames.forEach((cacheName) => {
              caches.delete(cacheName);
            });
          });
        }
      }
      localStorage.setItem('zlo_site_version', SITE_VERSION);
    };

    checkVersion();

    // Enregistrer le Service Worker
    if ('serviceWorker' in navigator) {
      navigator.serviceWorker
        .register('/sw.js')
        .then((registration) => {
          console.log('[SW] Registered:', registration);

          // Vérifier si une mise à jour est disponible
          registration.addEventListener('updatefound', () => {
            const newWorker = registration.installing;
            if (newWorker) {
              newWorker.addEventListener('statechange', () => {
                if (newWorker.state === 'installed' && navigator.serviceWorker.controller) {
                  console.log('[SW] Update available');
                  setIsUpdateAvailable(true);
                  setWaitingWorker(newWorker);
                  toast.info('Une mise à jour est disponible. Cliquez pour actualiser.', {
                    action: {
                      label: 'Actualiser',
                      onClick: () => {
                        newWorker.postMessage('SKIP_WAITING');
                        window.location.reload();
                      },
                    },
                    duration: 10000,
                  });
                }
              });
            }
          });

          // Vérifier si le site est prêt pour le offline
          if (registration.active) {
            setIsOfflineReady(true);
          }
        })
        .catch((error) => {
          console.error('[SW] Registration failed:', error);
        });

      // Écouter les messages du Service Worker
      navigator.serviceWorker.addEventListener('message', (event) => {
        if (event.data === 'UPDATE_AVAILABLE') {
          setIsUpdateAvailable(true);
        }
      });
    }

    // Vérifier la connexion internet
    const handleOnline = () => {
      toast.success('Connexion rétablie');
    };

    const handleOffline = () => {
      toast.warning('Vous êtes hors ligne. Certaines fonctionnalités peuvent être limitées.');
    };

    window.addEventListener('online', handleOnline);
    window.addEventListener('offline', handleOffline);

    return () => {
      window.removeEventListener('online', handleOnline);
      window.removeEventListener('offline', handleOffline);
    };
  }, []);

  return {
    isUpdateAvailable,
    isOfflineReady,
    updateApp,
  };
}

// Fonction pour forcer le rechargement du cache
export function forceCacheReload(): void {
  // Supprimer tous les caches
  if (typeof window !== 'undefined' && 'caches' in window) {
    caches.keys().then((cacheNames) => {
      Promise.all(cacheNames.map((cacheName) => caches.delete(cacheName)))
        .then(() => {
          console.log('[Cache] All caches cleared');
          (window as Window).location.reload();
        });
    });
  } else if (typeof window !== 'undefined') {
    (window as Window).location.reload();
  }
}

// Fonction pour vérifier si une nouvelle version est disponible
export async function checkForUpdates(): Promise<boolean> {
  try {
    const response = await fetch('/version.json', { 
      cache: 'no-store',
      headers: { 'Cache-Control': 'no-cache' }
    });
    if (response.ok) {
      const data = await response.json();
      return data.version !== SITE_VERSION;
    }
  } catch (error) {
    console.error('[Version] Check failed:', error);
  }
  return false;
}
