import { X, ShoppingCart, Check, Leaf, Droplets, Pill, Info } from 'lucide-react';
import { useFBOContent } from '../../context/FBOContentContext';
import { toast } from 'sonner';

interface FBOProductModalProps {
  productId: string | null;
  onClose: () => void;
}

export default function FBOProductModal({ productId, onClose }: FBOProductModalProps) {
  const { getProductById, addToCart } = useFBOContent();
  const product = productId ? getProductById(productId) : null;

  if (!product || !productId) return null;

  const handleAddToCart = () => {
    addToCart(product);
    toast.success(`${product.name} ajouté au panier`, {
      description: `${product.price.toFixed(2)} €`,
      icon: <Check className="w-4 h-4" />,
    });
    onClose();
  };

  return (
    <>
      {/* Backdrop */}
      <div
        className="fixed inset-0 bg-black/60 backdrop-blur-sm z-50"
        onClick={onClose}
      />

      {/* Modal */}
      <div className="fixed inset-4 sm:inset-10 lg:inset-20 bg-white rounded-2xl shadow-2xl z-50 overflow-hidden flex flex-col">
        {/* Header */}
        <div className="flex items-center justify-between p-6 border-b">
          <div className="flex items-center gap-3">
            <div className="w-10 h-10 rounded-full bg-[#e8f5e9] flex items-center justify-center">
              <Leaf className="w-5 h-5 text-[#2d7d32]" />
            </div>
            <div>
              <h2 className="text-xl font-bold text-[#1b5e20]">Détail du produit</h2>
              <p className="text-sm text-gray-500">Réf: {product.reference}</p>
            </div>
          </div>
          <button
            onClick={onClose}
            className="p-2 hover:bg-gray-100 rounded-full transition-colors"
          >
            <X className="w-6 h-6 text-gray-500" />
          </button>
        </div>

        {/* Content */}
        <div className="flex-1 overflow-y-auto">
          <div className="grid lg:grid-cols-2 gap-8 p-6 lg:p-10">
            {/* Left: Image */}
            <div className="relative">
              <div className="aspect-square rounded-2xl bg-gradient-to-br from-[#e8f5e9] to-[#c8e6c9] flex items-center justify-center overflow-hidden">
                <div className="w-40 h-40 rounded-full bg-white/50 flex items-center justify-center">
                  <span className="text-8xl">🌿</span>
                </div>
              </div>

              {/* Featured badge */}
              {product.featured && (
                <div className="absolute top-4 left-4 px-4 py-2 bg-gradient-to-r from-[#ff9800] to-[#ffc107] text-white font-bold rounded-full shadow-lg">
                  ⭐ Produit vedette
                </div>
              )}

              {/* Stock badge */}
              <div className="absolute bottom-4 left-4 px-4 py-2 bg-white/90 backdrop-blur-sm rounded-full shadow-md">
                <span className="flex items-center gap-2 text-sm font-medium text-[#2d7d32]">
                  <Check className="w-4 h-4" />
                  En stock
                </span>
              </div>
            </div>

            {/* Right: Details */}
            <div className="space-y-6">
              {/* Name & Price */}
              <div>
                <h1 className="text-3xl lg:text-4xl font-bold text-[#1b5e20] mb-4">
                  {product.name}
                </h1>
                <div className="flex items-baseline gap-3">
                  <span className="text-4xl font-bold text-[#2d7d32]">
                    {product.price.toFixed(2)} €
                  </span>
                  <span className="text-gray-400">TTC</span>
                </div>
              </div>

              {/* Description */}
              <div className="p-4 bg-[#f8fdf6] rounded-xl">
                <h3 className="font-semibold text-[#1b5e20] mb-2 flex items-center gap-2">
                  <Info className="w-5 h-5" />
                  Description
                </h3>
                <p className="text-gray-600 leading-relaxed">
                  {product.description}
                </p>
              </div>

              {/* Benefits */}
              <div>
                <h3 className="font-semibold text-[#1b5e20] mb-3 flex items-center gap-2">
                  <Check className="w-5 h-5" />
                  Bénéfices
                </h3>
                <ul className="grid sm:grid-cols-2 gap-2">
                  {product.benefits.map((benefit, index) => (
                    <li
                      key={index}
                      className="flex items-center gap-2 text-gray-600"
                    >
                      <span className="w-5 h-5 rounded-full bg-[#e8f5e9] flex items-center justify-center flex-shrink-0">
                        <Check className="w-3 h-3 text-[#2d7d32]" />
                      </span>
                      {benefit}
                    </li>
                  ))}
                </ul>
              </div>

              {/* Composition */}
              <div className="p-4 bg-gray-50 rounded-xl">
                <h3 className="font-semibold text-[#1b5e20] mb-2 flex items-center gap-2">
                  <Droplets className="w-5 h-5" />
                  Composition
                </h3>
                <p className="text-gray-600 text-sm leading-relaxed">
                  {product.composition}
                </p>
              </div>

              {/* Dosage */}
              <div className="p-4 bg-[#fff3e0] rounded-xl">
                <h3 className="font-semibold text-[#e65100] mb-2 flex items-center gap-2">
                  <Pill className="w-5 h-5" />
                  Posologie
                </h3>
                <p className="text-gray-700 text-sm leading-relaxed">
                  {product.dosage}
                </p>
              </div>

              {/* Add to Cart */}
              <button
                onClick={handleAddToCart}
                className="w-full py-4 bg-gradient-to-r from-[#2d7d32] to-[#4caf50] text-white font-semibold rounded-xl hover:shadow-xl hover:scale-[1.02] transition-all flex items-center justify-center gap-3"
              >
                <ShoppingCart className="w-6 h-6" />
                Ajouter au panier
                <span className="px-3 py-1 bg-white/20 rounded-full">
                  {product.price.toFixed(2)} €
                </span>
              </button>
            </div>
          </div>
        </div>
      </div>
    </>
  );
}
