import { X, Plus, Minus, ShoppingBag, Trash2, ArrowRight } from 'lucide-react';
import { useFBOContent } from '../../context/FBOContentContext';
import { toast } from 'sonner';

interface FBOCartProps {
  isOpen: boolean;
  onClose: () => void;
}

export default function FBOCart({ isOpen, onClose }: FBOCartProps) {
  const { cart, removeFromCart, updateCartQuantity, getCartTotal, clearCart } = useFBOContent();

  const handleCheckout = () => {
    toast.info('Fonctionnalité de commande en cours de développement', {
      description: 'Contactez-nous directement pour passer commande.',
    });
  };

  if (!isOpen) return null;

  return (
    <>
      {/* Backdrop */}
      <div
        className="fixed inset-0 bg-black/50 backdrop-blur-sm z-50"
        onClick={onClose}
      />

      {/* Cart Panel */}
      <div className="fixed right-0 top-0 h-full w-full max-w-md bg-white shadow-2xl z-50 flex flex-col">
        {/* Header */}
        <div className="flex items-center justify-between p-6 border-b">
          <div className="flex items-center gap-3">
            <ShoppingBag className="w-6 h-6 text-[#2d7d32]" />
            <h2 className="text-xl font-bold text-[#1b5e20]">Votre panier</h2>
            <span className="px-3 py-1 bg-[#e8f5e9] text-[#2d7d32] text-sm font-medium rounded-full">
              {cart.length} article{cart.length > 1 ? 's' : ''}
            </span>
          </div>
          <button
            onClick={onClose}
            className="p-2 hover:bg-gray-100 rounded-full transition-colors"
          >
            <X className="w-6 h-6 text-gray-500" />
          </button>
        </div>

        {/* Cart Items */}
        <div className="flex-1 overflow-y-auto p-6">
          {cart.length === 0 ? (
            <div className="flex flex-col items-center justify-center h-full text-center">
              <div className="w-24 h-24 rounded-full bg-[#e8f5e9] flex items-center justify-center mb-4">
                <ShoppingBag className="w-12 h-12 text-[#4caf50]" />
              </div>
              <h3 className="text-xl font-semibold text-gray-700 mb-2">
                Votre panier est vide
              </h3>
              <p className="text-gray-500 mb-6">
                Découvrez nos produits et ajoutez-les à votre panier
              </p>
              <button
                onClick={onClose}
                className="px-6 py-3 bg-gradient-to-r from-[#2d7d32] to-[#4caf50] text-white font-medium rounded-full hover:shadow-lg transition-shadow"
              >
                Explorer le catalogue
              </button>
            </div>
          ) : (
            <div className="space-y-4">
              {cart.map((item) => (
                <div
                  key={item.id}
                  className="flex gap-4 p-4 bg-gray-50 rounded-xl"
                >
                  {/* Image */}
                  <div className="w-20 h-20 rounded-lg bg-gradient-to-br from-[#e8f5e9] to-[#c8e6c9] flex items-center justify-center flex-shrink-0">
                    <span className="text-2xl">🌿</span>
                  </div>

                  {/* Content */}
                  <div className="flex-1 min-w-0">
                    <h4 className="font-semibold text-[#1b5e20] truncate">
                      {item.name}
                    </h4>
                    <p className="text-sm text-gray-500 mb-2">
                      Réf: {item.reference}
                    </p>
                    <div className="flex items-center justify-between">
                      {/* Quantity Controls */}
                      <div className="flex items-center gap-2">
                        <button
                          onClick={() => updateCartQuantity(item.id, item.quantity - 1)}
                          className="w-8 h-8 rounded-full bg-white border border-gray-200 flex items-center justify-center hover:bg-gray-100 transition-colors"
                        >
                          <Minus className="w-4 h-4" />
                        </button>
                        <span className="w-8 text-center font-medium">
                          {item.quantity}
                        </span>
                        <button
                          onClick={() => updateCartQuantity(item.id, item.quantity + 1)}
                          className="w-8 h-8 rounded-full bg-white border border-gray-200 flex items-center justify-center hover:bg-gray-100 transition-colors"
                        >
                          <Plus className="w-4 h-4" />
                        </button>
                      </div>

                      {/* Price */}
                      <span className="font-bold text-[#2d7d32]">
                        {(item.price * item.quantity).toFixed(2)} €
                      </span>
                    </div>
                  </div>

                  {/* Remove Button */}
                  <button
                    onClick={() => removeFromCart(item.id)}
                    className="p-2 text-gray-400 hover:text-red-500 transition-colors"
                  >
                    <Trash2 className="w-5 h-5" />
                  </button>
                </div>
              ))}
            </div>
          )}
        </div>

        {/* Footer */}
        {cart.length > 0 && (
          <div className="border-t p-6 space-y-4">
            {/* Clear Cart */}
            <button
              onClick={clearCart}
              className="text-sm text-gray-500 hover:text-red-500 transition-colors flex items-center gap-2"
            >
              <Trash2 className="w-4 h-4" />
              Vider le panier
            </button>

            {/* Total */}
            <div className="flex items-center justify-between py-4 border-t">
              <span className="text-gray-600">Total</span>
              <span className="text-2xl font-bold text-[#1b5e20]">
                {getCartTotal().toFixed(2)} €
              </span>
            </div>

            {/* Checkout Button */}
            <button
              onClick={handleCheckout}
              className="w-full py-4 bg-gradient-to-r from-[#2d7d32] to-[#4caf50] text-white font-semibold rounded-xl hover:shadow-lg hover:scale-[1.02] transition-all flex items-center justify-center gap-2"
            >
              Passer commande
              <ArrowRight className="w-5 h-5" />
            </button>

            <p className="text-xs text-gray-400 text-center">
              Les frais de livraison seront calculés lors de la commande
            </p>
          </div>
        )}
      </div>
    </>
  );
}
