import { useState } from 'react';
import { Leaf, ShoppingBag, Phone, Palette, Settings } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Label } from '@/components/ui/label';
import { Switch } from '@/components/ui/switch';
import { useFBOAdmin } from '@/context/FBOAdminContext';
import { useToast } from '@/hooks/use-toast';

// Product Editor Component
function ProductEditor() {
  const { products, addProduct, deleteProduct, toggleProductStock, toggleProductFeatured } = useFBOAdmin();
  const { toast } = useToast();

  return (
    <div className="space-y-4">
      <div className="flex justify-between items-center">
        <h3 className="text-lg font-semibold text-white">Produits</h3>
        <Button 
          onClick={() => {
            addProduct({
              reference: 'NEW',
              name: 'Nouveau produit',
              nameEn: 'New product',
              category: 'aloe-vera',
              price: 0,
              priceEUR: 0,
              priceUSD: 0,
              priceGBP: 0,
              image: 'https://via.placeholder.com/400',
              images: [],
              description: 'Description du produit',
              descriptionEn: 'Product description',
              composition: '',
              compositionEn: '',
              dosage: '',
              dosageEn: '',
              benefits: [],
              benefitsEn: [],
              inStock: true,
              featured: false
            });
            toast({ title: 'Produit créé', description: 'Le nouveau produit a été ajouté.' });
          }}
          className="bg-red-500 hover:bg-red-600"
        >
          + Ajouter un produit
        </Button>
      </div>

      <div className="grid gap-4">
        {products.map((product) => (
          <div key={product.id} className="p-4 bg-[#1a1a1f] border border-white/10 rounded-lg">
            <div className="flex items-center gap-4">
              <img src={product.image} alt={product.name} className="w-16 h-16 object-cover rounded-lg" />
              <div className="flex-1">
                <h4 className="text-white font-medium">{product.name}</h4>
                <p className="text-white/50 text-sm">{product.reference} - {product.price} FCFA</p>
              </div>
              <div className="flex items-center gap-2">
                <Switch
                  checked={product.inStock}
                  onCheckedChange={() => toggleProductStock(product.id)}
                />
                <span className="text-white/60 text-sm">Stock</span>
              </div>
              <div className="flex items-center gap-2">
                <Switch
                  checked={product.featured}
                  onCheckedChange={() => toggleProductFeatured(product.id)}
                />
                <span className="text-white/60 text-sm">Vedette</span>
              </div>
              <Button
                variant="destructive"
                size="sm"
                onClick={() => {
                  deleteProduct(product.id);
                  toast({ title: 'Produit supprimé' });
                }}
              >
                Supprimer
              </Button>
            </div>
          </div>
        ))}
      </div>
    </div>
  );
}

// Hero Editor Component
function HeroEditor() {
  const { hero, updateHero } = useFBOAdmin();

  return (
    <div className="space-y-4">
      <h3 className="text-lg font-semibold text-white">Hero Section</h3>
      <div className="p-4 bg-[#1a1a1f] border border-white/10 rounded-lg space-y-4">
        <div>
          <Label className="text-white">Titre (FR)</Label>
          <Input
            value={hero.title}
            onChange={(e) => updateHero({ title: e.target.value })}
            className="bg-white/5 border-white/10 text-white mt-1"
          />
        </div>
        <div>
          <Label className="text-white">Titre (EN)</Label>
          <Input
            value={hero.titleEn}
            onChange={(e) => updateHero({ titleEn: e.target.value })}
            className="bg-white/5 border-white/10 text-white mt-1"
          />
        </div>
        <div>
          <Label className="text-white">Description (FR)</Label>
          <Textarea
            value={hero.description}
            onChange={(e) => updateHero({ description: e.target.value })}
            className="bg-white/5 border-white/10 text-white mt-1"
            rows={3}
          />
        </div>
        <div>
          <Label className="text-white">Image de fond (URL)</Label>
          <Input
            value={hero.backgroundImage}
            onChange={(e) => updateHero({ backgroundImage: e.target.value })}
            className="bg-white/5 border-white/10 text-white mt-1"
          />
        </div>
      </div>
    </div>
  );
}

// Contact Editor Component
function ContactEditor() {
  const { contact, updateContact } = useFBOAdmin();

  return (
    <div className="space-y-4">
      <h3 className="text-lg font-semibold text-white">Contact FBO</h3>
      <div className="p-4 bg-[#1a1a1f] border border-white/10 rounded-lg space-y-4">
        <div className="grid sm:grid-cols-2 gap-4">
          <div>
            <Label className="text-white">Téléphone</Label>
            <Input
              value={contact.phone}
              onChange={(e) => updateContact({ phone: e.target.value })}
              className="bg-white/5 border-white/10 text-white mt-1"
            />
          </div>
          <div>
            <Label className="text-white">WhatsApp</Label>
            <Input
              value={contact.whatsapp}
              onChange={(e) => updateContact({ whatsapp: e.target.value })}
              className="bg-white/5 border-white/10 text-white mt-1"
            />
          </div>
        </div>
        <div>
          <Label className="text-white">Email</Label>
          <Input
            value={contact.email}
            onChange={(e) => updateContact({ email: e.target.value })}
            className="bg-white/5 border-white/10 text-white mt-1"
          />
        </div>
        <div>
          <Label className="text-white">Adresse</Label>
          <Input
            value={contact.address}
            onChange={(e) => updateContact({ address: e.target.value })}
            className="bg-white/5 border-white/10 text-white mt-1"
          />
        </div>
      </div>
    </div>
  );
}

// Theme Editor Component
function ThemeEditor() {
  const { theme, updateTheme, resetTheme } = useFBOAdmin();

  return (
    <div className="space-y-4">
      <div className="flex justify-between items-center">
        <h3 className="text-lg font-semibold text-white">Thème visuel</h3>
        <Button variant="outline" onClick={resetTheme} className="border-white/10 text-white hover:bg-white/10">
          Réinitialiser
        </Button>
      </div>
      <div className="p-4 bg-[#1a1a1f] border border-white/10 rounded-lg space-y-4">
        <div className="grid sm:grid-cols-2 gap-4">
          <div>
            <Label className="text-white">Couleur principale</Label>
            <div className="flex gap-2 mt-1">
              <input
                type="color"
                value={theme.primaryColor}
                onChange={(e) => updateTheme({ primaryColor: e.target.value })}
                className="w-12 h-10 rounded cursor-pointer"
              />
              <Input
                value={theme.primaryColor}
                onChange={(e) => updateTheme({ primaryColor: e.target.value })}
                className="flex-1 bg-white/5 border-white/10 text-white"
              />
            </div>
          </div>
          <div>
            <Label className="text-white">Couleur secondaire</Label>
            <div className="flex gap-2 mt-1">
              <input
                type="color"
                value={theme.secondaryColor}
                onChange={(e) => updateTheme({ secondaryColor: e.target.value })}
                className="w-12 h-10 rounded cursor-pointer"
              />
              <Input
                value={theme.secondaryColor}
                onChange={(e) => updateTheme({ secondaryColor: e.target.value })}
                className="flex-1 bg-white/5 border-white/10 text-white"
              />
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}

// Settings Component
function SettingsEditor() {
  const { showFBOButton, showFBOMenu, setShowFBOButton, setShowFBOMenu } = useFBOAdmin();

  return (
    <div className="space-y-4">
      <h3 className="text-lg font-semibold text-white">Paramètres FBO</h3>
      <div className="p-4 bg-[#1a1a1f] border border-white/10 rounded-lg space-y-4">
        <div className="flex items-center justify-between">
          <div>
            <Label className="text-white">Afficher le bouton FBO</Label>
            <p className="text-white/50 text-sm">Bouton flottant sur le site principal</p>
          </div>
          <Switch checked={showFBOButton} onCheckedChange={setShowFBOButton} />
        </div>
        <div className="flex items-center justify-between">
          <div>
            <Label className="text-white">Afficher FBO dans le menu</Label>
            <p className="text-white/50 text-sm">Lien FBO dans la navigation</p>
          </div>
          <Switch checked={showFBOMenu} onCheckedChange={setShowFBOMenu} />
        </div>
      </div>
    </div>
  );
}

// Main FBO Editor
export default function FBOEditor() {
  const [activeTab, setActiveTab] = useState('products');

  return (
    <div className="space-y-6">
      {/* Header */}
      <div>
        <h1 className="text-2xl font-bold text-white">Gestion FBO</h1>
        <p className="text-white/60">Gérez votre sous-site Forever Living Products</p>
      </div>

      {/* Tabs */}
      <Tabs value={activeTab} onValueChange={setActiveTab}>
        <TabsList className="bg-[#141419] border border-white/10 flex-wrap h-auto">
          <TabsTrigger value="products" className="data-[state=active]:bg-red-500 data-[state=active]:text-white">
            <ShoppingBag className="w-4 h-4 mr-2" />
            Produits
          </TabsTrigger>
          <TabsTrigger value="hero" className="data-[state=active]:bg-red-500 data-[state=active]:text-white">
            <Leaf className="w-4 h-4 mr-2" />
            Hero
          </TabsTrigger>
          <TabsTrigger value="contact" className="data-[state=active]:bg-red-500 data-[state=active]:text-white">
            <Phone className="w-4 h-4 mr-2" />
            Contact
          </TabsTrigger>
          <TabsTrigger value="theme" className="data-[state=active]:bg-red-500 data-[state=active]:text-white">
            <Palette className="w-4 h-4 mr-2" />
            Thème
          </TabsTrigger>
          <TabsTrigger value="settings" className="data-[state=active]:bg-red-500 data-[state=active]:text-white">
            <Settings className="w-4 h-4 mr-2" />
            Paramètres
          </TabsTrigger>
        </TabsList>

        <TabsContent value="products" className="mt-6">
          <ProductEditor />
        </TabsContent>

        <TabsContent value="hero" className="mt-6">
          <HeroEditor />
        </TabsContent>

        <TabsContent value="contact" className="mt-6">
          <ContactEditor />
        </TabsContent>

        <TabsContent value="theme" className="mt-6">
          <ThemeEditor />
        </TabsContent>

        <TabsContent value="settings" className="mt-6">
          <SettingsEditor />
        </TabsContent>
      </Tabs>

      {/* Preview Link */}
      <div className="p-4 bg-blue-500/10 border border-blue-500/20 rounded-lg">
        <a 
          href="/fbo" 
          target="_blank" 
          className="flex items-center text-blue-400 hover:text-blue-300"
        >
          Voir le site FBO
        </a>
      </div>
    </div>
  );
}
