import { useEffect, useState } from 'react';
import { 
  Eye, 
  MousePointer, 
  ShoppingCart, 
  Users,
  ArrowUpRight,
  ArrowDownRight
} from 'lucide-react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { useContent } from '@/context/ContentContext';
import { useFBOAdmin } from '@/context/FBOAdminContext';

interface StatCardProps {
  title: string;
  value: string;
  change: string;
  trend: 'up' | 'down';
  icon: React.ElementType;
}

function StatCard({ title, value, change, trend, icon: Icon }: StatCardProps) {
  return (
    <Card className="bg-[#141419] border-white/10">
      <CardHeader className="flex flex-row items-center justify-between pb-2">
        <CardTitle className="text-sm font-medium text-white/60">{title}</CardTitle>
        <div className="w-8 h-8 rounded-lg bg-red-500/10 flex items-center justify-center">
          <Icon className="w-4 h-4 text-red-500" />
        </div>
      </CardHeader>
      <CardContent>
        <div className="text-2xl font-bold text-white">{value}</div>
        <div className={`flex items-center gap-1 text-sm ${
          trend === 'up' ? 'text-green-400' : 'text-red-400'
        }`}>
          {trend === 'up' ? <ArrowUpRight className="w-4 h-4" /> : <ArrowDownRight className="w-4 h-4" />}
          <span>{change}</span>
        </div>
      </CardContent>
    </Card>
  );
}

export default function Dashboard() {
  const { content } = useContent();
  const { products, cart } = useFBOAdmin();
  const [stats, setStats] = useState({
    visits: '0',
    pageViews: '0',
    orders: '0',
    users: '0'
  });

  useEffect(() => {
    // Simulate loading stats from localStorage or API
    const savedVisits = localStorage.getItem('zlo_visits') || '1,234';
    setStats({
      visits: savedVisits,
      pageViews: '5,678',
      orders: cart.length.toString(),
      users: content.team.length.toString()
    });
  }, [cart, content.team.length]);

  return (
    <div className="space-y-6">
      {/* Header */}
      <div>
        <h1 className="text-2xl font-bold text-white">Tableau de bord</h1>
        <p className="text-white/60">Bienvenue dans votre panneau d'administration</p>
      </div>

      {/* Stats Grid */}
      <div className="grid sm:grid-cols-2 lg:grid-cols-4 gap-4">
        <StatCard
          title="Visites"
          value={stats.visits}
          change="+12% ce mois"
          trend="up"
          icon={Eye}
        />
        <StatCard
          title="Pages vues"
          value={stats.pageViews}
          change="+8% ce mois"
          trend="up"
          icon={MousePointer}
        />
        <StatCard
          title="Commandes"
          value={stats.orders}
          change="+5% ce mois"
          trend="up"
          icon={ShoppingCart}
        />
        <StatCard
          title="Membres équipe"
          value={stats.users}
          change="Stable"
          trend="up"
          icon={Users}
        />
      </div>

      {/* Content Overview */}
      <div className="grid lg:grid-cols-2 gap-6">
        {/* Services */}
        <Card className="bg-[#141419] border-white/10">
          <CardHeader>
            <CardTitle className="text-white">Services</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {content.services.slice(0, 5).map((service) => (
                <div 
                  key={service.id}
                  className="flex items-center justify-between p-3 rounded-lg bg-white/5"
                >
                  <span className="text-white/80">{service.title}</span>
                  <span className="text-white/40 text-sm">{service.shortDescription.slice(0, 30)}...</span>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>

        {/* FBO Products */}
        <Card className="bg-[#141419] border-white/10">
          <CardHeader>
            <CardTitle className="text-white">Produits FBO</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {products.slice(0, 5).map((product) => (
                <div 
                  key={product.id}
                  className="flex items-center justify-between p-3 rounded-lg bg-white/5"
                >
                  <div className="flex items-center gap-3">
                    <img 
                      src={product.image} 
                      alt={product.name}
                      className="w-10 h-10 rounded object-cover"
                    />
                    <span className="text-white/80">{product.name}</span>
                  </div>
                  <span className="text-red-400 font-medium">{product.price} FCFA</span>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Quick Actions */}
      <Card className="bg-[#141419] border-white/10">
        <CardHeader>
          <CardTitle className="text-white">Actions rapides</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex flex-wrap gap-3">
            <a 
              href="/admin/content"
              className="px-4 py-2 bg-red-500/10 text-red-400 rounded-lg hover:bg-red-500/20 transition-colors"
            >
              Modifier le contenu
            </a>
            <a 
              href="/admin/sections"
              className="px-4 py-2 bg-red-500/10 text-red-400 rounded-lg hover:bg-red-500/20 transition-colors"
            >
              Gérer les sections
            </a>
            <a 
              href="/admin/fbo"
              className="px-4 py-2 bg-red-500/10 text-red-400 rounded-lg hover:bg-red-500/20 transition-colors"
            >
              Gérer les produits FBO
            </a>
            <a 
              href="/"
              target="_blank"
              className="px-4 py-2 bg-white/5 text-white/70 rounded-lg hover:bg-white/10 transition-colors"
            >
              Voir le site
            </a>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
