import { useState } from 'react';
import { RefreshCw, Download, Upload } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Textarea } from '@/components/ui/textarea';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { useContent } from '@/context/ContentContext';
import { useToast } from '@/hooks/use-toast';

export default function ContentEditor() {
  const { 
    content, 
    updateHero, 
    updateAbout, 
    updateContact, 
    exportContent, 
    importContent,
    resetContent 
  } = useContent();
  const { toast } = useToast();
  const [activeTab, setActiveTab] = useState('hero');
  const [importData, setImportData] = useState('');

  const handleExport = () => {
    const data = exportContent();
    const blob = new Blob([data], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'zlo-content.json';
    a.click();
    URL.revokeObjectURL(url);
    
    toast({
      title: 'Export réussi',
      description: 'Le contenu a été exporté avec succès.',
    });
  };

  const handleImport = () => {
    try {
      importContent(importData);
      setImportData('');
      toast({
        title: 'Import réussi',
        description: 'Le contenu a été importé avec succès.',
      });
    } catch {
      toast({
        title: 'Erreur',
        description: 'Le format du fichier est invalide.',
        variant: 'destructive'
      });
    }
  };

  const handleReset = () => {
    if (confirm('Êtes-vous sûr de vouloir réinitialiser tout le contenu ?')) {
      resetContent();
      toast({
        title: 'Réinitialisation',
        description: 'Le contenu a été réinitialisé.',
      });
    }
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold text-white">Éditeur de contenu</h1>
          <p className="text-white/60">Modifiez le contenu de votre site</p>
        </div>
        <div className="flex gap-2">
          <Button variant="outline" onClick={handleExport} className="border-white/10 text-white hover:bg-white/10">
            <Download className="w-4 h-4 mr-2" />
            Exporter
          </Button>
          <Button variant="outline" onClick={handleReset} className="border-red-500/30 text-red-400 hover:bg-red-500/10">
            <RefreshCw className="w-4 h-4 mr-2" />
            Réinitialiser
          </Button>
        </div>
      </div>

      {/* Import Section */}
      <div className="p-4 bg-[#141419] border border-white/10 rounded-lg">
        <Label className="text-white mb-2 block">Importer du contenu (JSON)</Label>
        <div className="flex gap-2">
          <Textarea
            value={importData}
            onChange={(e) => setImportData(e.target.value)}
            placeholder="Collez le contenu JSON ici..."
            className="flex-1 bg-white/5 border-white/10 text-white placeholder:text-white/30"
          />
          <Button onClick={handleImport} className="bg-red-500 hover:bg-red-600">
            <Upload className="w-4 h-4 mr-2" />
            Importer
          </Button>
        </div>
      </div>

      {/* Tabs */}
      <Tabs value={activeTab} onValueChange={setActiveTab}>
        <TabsList className="bg-[#141419] border border-white/10">
          <TabsTrigger value="hero" className="data-[state=active]:bg-red-500 data-[state=active]:text-white">Hero</TabsTrigger>
          <TabsTrigger value="about" className="data-[state=active]:bg-red-500 data-[state=active]:text-white">À propos</TabsTrigger>
          <TabsTrigger value="contact" className="data-[state=active]:bg-red-500 data-[state=active]:text-white">Contact</TabsTrigger>
        </TabsList>

        {/* Hero Tab */}
        <TabsContent value="hero" className="space-y-4">
          <div className="p-6 bg-[#141419] border border-white/10 rounded-lg space-y-4">
            <div>
              <Label className="text-white">Titre</Label>
              <Input
                value={content.hero.title}
                onChange={(e) => updateHero({ title: e.target.value })}
                className="bg-white/5 border-white/10 text-white mt-1"
              />
            </div>
            <div>
              <Label className="text-white">Sous-titre</Label>
              <Input
                value={content.hero.subtitle}
                onChange={(e) => updateHero({ subtitle: e.target.value })}
                className="bg-white/5 border-white/10 text-white mt-1"
              />
            </div>
            <div>
              <Label className="text-white">Description</Label>
              <Textarea
                value={content.hero.description}
                onChange={(e) => updateHero({ description: e.target.value })}
                className="bg-white/5 border-white/10 text-white mt-1"
                rows={3}
              />
            </div>
            <div className="grid sm:grid-cols-2 gap-4">
              <div>
                <Label className="text-white">CTA Principal</Label>
                <Input
                  value={content.hero.ctaPrimary}
                  onChange={(e) => updateHero({ ctaPrimary: e.target.value })}
                  className="bg-white/5 border-white/10 text-white mt-1"
                />
              </div>
              <div>
                <Label className="text-white">CTA Secondaire</Label>
                <Input
                  value={content.hero.ctaSecondary}
                  onChange={(e) => updateHero({ ctaSecondary: e.target.value })}
                  className="bg-white/5 border-white/10 text-white mt-1"
                />
              </div>
            </div>
            <div>
              <Label className="text-white">Image de fond (URL)</Label>
              <Input
                value={content.hero.backgroundImage}
                onChange={(e) => updateHero({ backgroundImage: e.target.value })}
                className="bg-white/5 border-white/10 text-white mt-1"
              />
            </div>
          </div>
        </TabsContent>

        {/* About Tab */}
        <TabsContent value="about" className="space-y-4">
          <div className="p-6 bg-[#141419] border border-white/10 rounded-lg space-y-4">
            <div>
              <Label className="text-white">Titre</Label>
              <Input
                value={content.about.title}
                onChange={(e) => updateAbout({ title: e.target.value })}
                className="bg-white/5 border-white/10 text-white mt-1"
              />
            </div>
            <div>
              <Label className="text-white">Description</Label>
              <Textarea
                value={content.about.description}
                onChange={(e) => updateAbout({ description: e.target.value })}
                className="bg-white/5 border-white/10 text-white mt-1"
                rows={3}
              />
            </div>
            <div>
              <Label className="text-white">Mission</Label>
              <Textarea
                value={content.about.mission}
                onChange={(e) => updateAbout({ mission: e.target.value })}
                className="bg-white/5 border-white/10 text-white mt-1"
                rows={2}
              />
            </div>
            <div>
              <Label className="text-white">Vision</Label>
              <Textarea
                value={content.about.vision}
                onChange={(e) => updateAbout({ vision: e.target.value })}
                className="bg-white/5 border-white/10 text-white mt-1"
                rows={2}
              />
            </div>
            <div>
              <Label className="text-white">Image (URL)</Label>
              <Input
                value={content.about.image}
                onChange={(e) => updateAbout({ image: e.target.value })}
                className="bg-white/5 border-white/10 text-white mt-1"
              />
            </div>
          </div>
        </TabsContent>

        {/* Contact Tab */}
        <TabsContent value="contact" className="space-y-4">
          <div className="p-6 bg-[#141419] border border-white/10 rounded-lg space-y-4">
            <div>
              <Label className="text-white">Titre</Label>
              <Input
                value={content.contact.title}
                onChange={(e) => updateContact({ title: e.target.value })}
                className="bg-white/5 border-white/10 text-white mt-1"
              />
            </div>
            <div>
              <Label className="text-white">Sous-titre</Label>
              <Input
                value={content.contact.subtitle}
                onChange={(e) => updateContact({ subtitle: e.target.value })}
                className="bg-white/5 border-white/10 text-white mt-1"
              />
            </div>
            <div>
              <Label className="text-white">Description</Label>
              <Textarea
                value={content.contact.description}
                onChange={(e) => updateContact({ description: e.target.value })}
                className="bg-white/5 border-white/10 text-white mt-1"
                rows={2}
              />
            </div>
            <div className="grid sm:grid-cols-2 gap-4">
              <div>
                <Label className="text-white">Adresse</Label>
                <Input
                  value={content.contact.address}
                  onChange={(e) => updateContact({ address: e.target.value })}
                  className="bg-white/5 border-white/10 text-white mt-1"
                />
              </div>
              <div>
                <Label className="text-white">Téléphone</Label>
                <Input
                  value={content.contact.phone}
                  onChange={(e) => updateContact({ phone: e.target.value })}
                  className="bg-white/5 border-white/10 text-white mt-1"
                />
              </div>
            </div>
            <div className="grid sm:grid-cols-2 gap-4">
              <div>
                <Label className="text-white">Email</Label>
                <Input
                  value={content.contact.email}
                  onChange={(e) => updateContact({ email: e.target.value })}
                  className="bg-white/5 border-white/10 text-white mt-1"
                />
              </div>
              <div>
                <Label className="text-white">Horaires</Label>
                <Input
                  value={content.contact.hours}
                  onChange={(e) => updateContact({ hours: e.target.value })}
                  className="bg-white/5 border-white/10 text-white mt-1"
                />
              </div>
            </div>
          </div>
        </TabsContent>
      </Tabs>
    </div>
  );
}
