import { useEffect, useRef } from 'react';
import { useTranslation } from 'react-i18next';
import gsap from 'gsap';
import { ScrollTrigger } from 'gsap/ScrollTrigger';

gsap.registerPlugin(ScrollTrigger);

const Capabilities = () => {
  const { t } = useTranslation();
  const sectionRef = useRef<HTMLElement>(null);

  useEffect(() => {
    const section = sectionRef.current;
    if (!section) return;

    const ctx = gsap.context(() => {
      gsap.from('.cap-pill', {
        scrollTrigger: {
          trigger: section,
          start: 'top 70%',
          toggleActions: 'play none none reverse',
        },
        y: 60,
        opacity: 0,
        duration: 0.8,
        stagger: 0.15,
        ease: 'power3.out',
      });

      gsap.from('.cap-text', {
        scrollTrigger: {
          trigger: section,
          start: 'top 60%',
          toggleActions: 'play none none reverse',
        },
        x: 40,
        opacity: 0,
        duration: 0.8,
        stagger: 0.1,
        ease: 'power3.out',
      });
    }, section);

    return () => ctx.revert();
  }, []);

  return (
    <section
      ref={sectionRef}
      className="min-h-screen bg-zlo-dark grid-lines relative overflow-hidden py-24"
    >
      <div className="relative w-full px-6 lg:px-12">
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-center min-h-[80vh]">
          {/* Left - Pills Grid */}
          <div className="grid grid-cols-2 gap-4">
            <div className="cap-pill col-span-2 lg:col-span-1 aspect-[4/3] pill-frame">
              <div className="relative w-full h-full">
                <img
                  src="/images/mosaic_tech_workspace.jpg"
                  alt="Tech Workspace"
                  className="w-full h-full object-cover img-treated"
                />
                <div className="absolute inset-0 bg-gradient-to-t from-zlo-dark/60 to-transparent" />
                <div className="absolute top-4 left-4 px-3 py-1 bg-zlo-dark/80 backdrop-blur-sm rounded-full">
                  <span className="label-text text-zlo-orange text-[10px]">{t('capabilities.pill1')}</span>
                </div>
              </div>
            </div>
            <div className="cap-pill col-span-2 lg:col-span-1 aspect-[4/3] pill-frame">
              <div className="relative w-full h-full">
                <img
                  src="/images/mosaic_server_room.jpg"
                  alt="Server Room"
                  className="w-full h-full object-cover img-treated"
                />
                <div className="absolute inset-0 bg-gradient-to-t from-zlo-dark/60 to-transparent" />
                <div className="absolute top-4 left-4 px-3 py-1 bg-zlo-dark/80 backdrop-blur-sm rounded-full">
                  <span className="label-text text-zlo-orange text-[10px]">{t('capabilities.pill2')}</span>
                </div>
              </div>
            </div>
            <div className="cap-pill col-span-2 aspect-[16/9] lg:aspect-[2/1] pill-frame">
              <div className="relative w-full h-full">
                <img
                  src="/images/mosaic_control_room.jpg"
                  alt="Control Room"
                  className="w-full h-full object-cover img-treated"
                />
                <div className="absolute inset-0 bg-gradient-to-t from-zlo-dark/60 to-transparent" />
                <div className="absolute top-4 left-4 px-3 py-1 bg-zlo-dark/80 backdrop-blur-sm rounded-full">
                  <span className="label-text text-zlo-orange text-[10px]">{t('capabilities.pill3')}</span>
                </div>
              </div>
            </div>
          </div>

          {/* Right - Text Block */}
          <div className="lg:pl-12">
            <h2 className="cap-text heading-lg text-white mb-2">
              {t('capabilities.title1')}
            </h2>
            <h2 className="cap-text heading-lg gradient-text mb-8">
              {t('capabilities.title2')}
            </h2>
            <div className="cap-text gradient-line w-[60%] max-w-[300px] mb-8" />
            <p className="cap-text body-text text-zlo-text-secondary mb-8">
              {t('capabilities.body')}
            </p>
            <button className="cap-text cta-button-outline">
              {t('capabilities.cta')}
            </button>
          </div>
        </div>
      </div>
    </section>
  );
};

export default Capabilities;