import { useEffect, useRef } from 'react';
import { Leaf, Award, Heart, Truck } from 'lucide-react';
import { useFBOContent } from '../../context/FBOContentContext';
import gsap from 'gsap';
import { ScrollTrigger } from 'gsap/ScrollTrigger';

gsap.registerPlugin(ScrollTrigger);

const iconMap: Record<string, React.ElementType> = {
  Leaf,
  Award,
  Heart,
  Truck,
};

export default function FBOBenefits() {
  const { content } = useFBOContent();
  const sectionRef = useRef<HTMLDivElement>(null);
  const cardsRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const ctx = gsap.context(() => {
      const cards = cardsRef.current?.children;
      if (cards) {
        gsap.from(cards, {
          scrollTrigger: {
            trigger: sectionRef.current,
            start: 'top 80%',
            toggleActions: 'play none none reverse',
          },
          y: 60,
          opacity: 0,
          duration: 0.6,
          stagger: 0.15,
          ease: 'power3.out',
        });
      }
    }, sectionRef);

    return () => ctx.revert();
  }, []);

  return (
    <section
      id="benefits"
      ref={sectionRef}
      className="py-20 lg:py-32 bg-gradient-to-b from-transparent to-[#e8f5e9]/30"
    >
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <div className="text-center mb-16">
          <div className="inline-flex items-center gap-2 px-4 py-2 bg-[#e8f5e9] rounded-full mb-6">
            <Heart className="w-4 h-4 text-[#4caf50]" />
            <span className="text-sm font-medium text-[#2d7d32]">Nos engagements</span>
          </div>
          <h2 className="text-3xl sm:text-4xl lg:text-5xl font-bold text-[#1b5e20] mb-4">
            {content.benefits.title}
          </h2>
          <p className="text-lg text-gray-600 max-w-2xl mx-auto">
            Des produits de qualité supérieure, développés avec soin pour votre bien-être
          </p>
        </div>

        {/* Benefits Cards */}
        <div
          ref={cardsRef}
          className="grid sm:grid-cols-2 lg:grid-cols-4 gap-6"
        >
          {content.benefits.items.map((benefit, index) => {
            const Icon = iconMap[benefit.icon] || Heart;
            return (
              <div
                key={index}
                className="group relative bg-white rounded-2xl p-6 shadow-sm hover:shadow-xl transition-all duration-300 hover:-translate-y-2 overflow-hidden"
              >
                {/* Gradient overlay on hover */}
                <div className="absolute inset-0 bg-gradient-to-br from-[#4caf50]/5 to-[#81c784]/5 opacity-0 group-hover:opacity-100 transition-opacity" />
                
                {/* Content */}
                <div className="relative z-10">
                  {/* Icon */}
                  <div className="w-14 h-14 rounded-xl bg-gradient-to-br from-[#e8f5e9] to-[#c8e6c9] flex items-center justify-center mb-5 group-hover:scale-110 transition-transform">
                    <Icon className="w-7 h-7 text-[#2d7d32]" />
                  </div>

                  {/* Text */}
                  <h3 className="text-xl font-bold text-[#1b5e20] mb-3">
                    {benefit.title}
                  </h3>
                  <p className="text-gray-600 text-sm leading-relaxed">
                    {benefit.description}
                  </p>
                </div>

                {/* Decorative corner */}
                <div className="absolute -bottom-4 -right-4 w-20 h-20 bg-gradient-to-br from-[#4caf50]/10 to-transparent rounded-full opacity-0 group-hover:opacity-100 transition-opacity" />
              </div>
            );
          })}
        </div>
      </div>
    </section>
  );
}
