import { useEffect, useRef } from 'react';
import { Award, Globe, Heart, Users } from 'lucide-react';
import { useFBOContent } from '../../context/FBOContentContext';
import gsap from 'gsap';
import { ScrollTrigger } from 'gsap/ScrollTrigger';

gsap.registerPlugin(ScrollTrigger);

export default function FBOAbout() {
  const { content } = useFBOContent();
  const sectionRef = useRef<HTMLDivElement>(null);
  const contentRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const ctx = gsap.context(() => {
      gsap.from(contentRef.current, {
        scrollTrigger: {
          trigger: sectionRef.current,
          start: 'top 80%',
          toggleActions: 'play none none reverse',
        },
        y: 50,
        opacity: 0,
        duration: 0.8,
        ease: 'power3.out',
      });
    }, sectionRef);

    return () => ctx.revert();
  }, []);

  const features = [
    {
      icon: Globe,
      title: 'Présence Mondiale',
      description: 'Plus de 160 pays',
    },
    {
      icon: Award,
      title: 'Certifié IASC',
      description: 'Qualité garantie',
    },
    {
      icon: Heart,
      title: '100% Naturel',
      description: 'Sans produits chimiques',
    },
    {
      icon: Users,
      title: 'Communauté FBO',
      description: '10+ millions de membres',
    },
  ];

  return (
    <section
      id="about"
      ref={sectionRef}
      className="py-20 lg:py-32 relative overflow-hidden"
    >
      {/* Background decoration */}
      <div className="absolute top-0 right-0 w-1/3 h-full bg-gradient-to-l from-[#e8f5e9]/50 to-transparent" />
      
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative z-10">
        <div
          ref={contentRef}
          className="grid lg:grid-cols-2 gap-12 lg:gap-20 items-center"
        >
          {/* Left: Image/Visual */}
          <div className="relative">
            <div className="aspect-square max-w-lg mx-auto relative">
              {/* Main circle */}
              <div className="absolute inset-0 rounded-full bg-gradient-to-br from-[#c8e6c9] to-[#a5d6a7] opacity-50" />
              
              {/* Inner content */}
              <div className="absolute inset-8 rounded-full bg-white shadow-xl flex items-center justify-center overflow-hidden">
                <div className="text-center p-8">
                  <div className="w-24 h-24 mx-auto mb-6 rounded-full bg-gradient-to-br from-[#2d7d32] to-[#4caf50] flex items-center justify-center">
                    <Award className="w-12 h-12 text-white" />
                  </div>
                  <div className="text-5xl font-bold text-[#1b5e20] mb-2">40+</div>
                  <div className="text-gray-600">Années d'expertise</div>
                </div>
              </div>
              
              {/* Orbiting badges */}
              <div className="absolute top-0 left-1/2 -translate-x-1/2 -translate-y-4 bg-white rounded-full px-4 py-2 shadow-lg">
                <span className="text-sm font-semibold text-[#2d7d32]">Depuis 1978</span>
              </div>
              <div className="absolute bottom-8 right-0 translate-x-4 bg-white rounded-full px-4 py-2 shadow-lg">
                <span className="text-sm font-semibold text-[#4caf50]">160+ Pays</span>
              </div>
              <div className="absolute bottom-8 left-0 -translate-x-4 bg-white rounded-full px-4 py-2 shadow-lg">
                <span className="text-sm font-semibold text-[#66bb6a]">100% Pur</span>
              </div>
            </div>
          </div>

          {/* Right: Content */}
          <div>
            <div className="inline-flex items-center gap-2 px-4 py-2 bg-[#e8f5e9] rounded-full mb-6">
              <span className="w-2 h-2 bg-[#4caf50] rounded-full animate-pulse" />
              <span className="text-sm font-medium text-[#2d7d32]">À propos de nous</span>
            </div>

            <h2 className="text-3xl sm:text-4xl lg:text-5xl font-bold text-[#1b5e20] mb-6 leading-tight">
              {content.about.title}
            </h2>

            <p className="text-lg text-gray-600 mb-8 leading-relaxed">
              {content.about.body}
            </p>

            {/* Features Grid */}
            <div className="grid grid-cols-2 gap-4">
              {features.map((feature, index) => (
                <div
                  key={index}
                  className="group p-4 bg-white rounded-xl shadow-sm hover:shadow-md transition-shadow"
                >
                  <feature.icon className="w-8 h-8 text-[#4caf50] mb-3 group-hover:scale-110 transition-transform" />
                  <h3 className="font-semibold text-[#1b5e20] mb-1">
                    {feature.title}
                  </h3>
                  <p className="text-sm text-gray-500">{feature.description}</p>
                </div>
              ))}
            </div>
          </div>
        </div>
      </div>
    </section>
  );
}
