import { useState, useEffect } from 'react';
import { Menu, X, Phone, Leaf, ExternalLink, Globe } from 'lucide-react';
import { useFBOAdmin } from '@/context/FBOAdminContext';

interface HeaderProps {
  onNavigate?: (section: string) => void;
}

export default function Header({ onNavigate }: HeaderProps) {
  const [isScrolled, setIsScrolled] = useState(false);
  const [isMobileMenuOpen, setIsMobileMenuOpen] = useState(false);
  const { showFBOMenu } = useFBOAdmin();

  useEffect(() => {
    const handleScroll = () => {
      setIsScrolled(window.scrollY > 50);
    };
    window.addEventListener('scroll', handleScroll, { passive: true });
    return () => window.removeEventListener('scroll', handleScroll);
  }, []);

  const navItems = [
    { label: 'Accueil', href: 'hero' },
    { label: 'À propos', href: 'about' },
    { label: 'Services', href: 'services' },
    { label: 'Projets', href: 'projects' },
    { label: 'La Team', href: 'team' },
    { label: 'Contact', href: 'contact' },
  ];

  const handleNavClick = (href: string) => {
    if (onNavigate) {
      onNavigate(href);
    }
    setIsMobileMenuOpen(false);
  };

  return (
    <header
      className={`fixed top-0 left-0 right-0 z-50 transition-all duration-500 ${
        isScrolled
          ? 'bg-[#0a0a0a]/95 backdrop-blur-md border-b border-white/5'
          : 'bg-transparent'
      }`}
    >
      <div className="container-custom">
        <div className="flex items-center justify-between h-16 lg:h-20">
          {/* Logo */}
          <a 
            href="#hero" 
            onClick={(e) => { e.preventDefault(); handleNavClick('hero'); }}
            className="flex items-center gap-0 group"
          >
            <span className="text-xl lg:text-2xl font-bold text-white tracking-tight">
              ZLO
            </span>
            <span className="text-xl lg:text-2xl font-bold text-orange-500 tracking-tight">
              Technologies
            </span>
          </a>

          {/* Desktop Navigation */}
          <nav className="hidden lg:flex items-center gap-8">
            {navItems.map((item) => (
              <button
                key={item.href}
                onClick={() => handleNavClick(item.href)}
                className="text-sm text-white/70 hover:text-white transition-colors relative group"
              >
                {item.label}
                <span className="absolute -bottom-1 left-0 w-0 h-0.5 bg-gradient-to-r from-red-500 to-orange-500 group-hover:w-full transition-all duration-300" />
              </button>
            ))}
          </nav>

          {/* Right Side Buttons */}
          <div className="flex items-center gap-3">
            {/* Talk to an expert button */}
            <a
              href="tel:+22892605949"
              className="hidden md:flex items-center gap-2 px-4 py-2 bg-gradient-to-r from-red-600 to-orange-500 text-white text-sm font-medium rounded-full hover:shadow-lg hover:shadow-red-500/30 transition-all duration-300 hover:scale-105"
            >
              <Phone className="w-4 h-4" />
              <span>Talk to an expert</span>
            </a>

            {/* FBO Button */}
            {showFBOMenu && (
              <a
                href="/fbo.html"
                className="hidden sm:flex items-center gap-2 px-4 py-2 bg-gradient-to-r from-green-600 to-green-500 text-white text-sm font-medium rounded-full hover:shadow-lg hover:shadow-green-500/30 transition-all duration-300 hover:scale-105"
              >
                <Leaf className="w-4 h-4" />
                <span>FBO</span>
                <ExternalLink className="w-3 h-3" />
              </a>
            )}

            {/* Language Switcher */}
            <button className="hidden sm:flex items-center gap-1.5 px-3 py-2 text-white/70 hover:text-white border border-white/20 hover:border-white/40 rounded-full text-sm transition-all">
              <Globe className="w-4 h-4" />
              <span>EN</span>
            </button>

            {/* Mobile Menu Button */}
            <button
              onClick={() => setIsMobileMenuOpen(!isMobileMenuOpen)}
              className="lg:hidden p-2 text-white hover:bg-white/10 rounded-lg transition-colors"
            >
              {isMobileMenuOpen ? <X className="w-6 h-6" /> : <Menu className="w-6 h-6" />}
            </button>
          </div>
        </div>
      </div>

      {/* Mobile Menu */}
      {isMobileMenuOpen && (
        <div className="lg:hidden bg-[#0a0a0a]/98 backdrop-blur-md border-t border-white/10">
          <nav className="flex flex-col p-4 space-y-1">
            {navItems.map((item) => (
              <button
                key={item.href}
                onClick={() => handleNavClick(item.href)}
                className="text-left py-3 px-4 text-white/80 hover:text-white hover:bg-white/5 rounded-lg transition-colors"
              >
                {item.label}
              </button>
            ))}
            <div className="pt-4 border-t border-white/10 space-y-2">
              <a
                href="tel:+22892605949"
                className="flex items-center gap-2 px-4 py-3 bg-gradient-to-r from-red-600 to-orange-500 text-white rounded-lg"
              >
                <Phone className="w-5 h-5" />
                <span>Talk to an expert</span>
              </a>
              {showFBOMenu && (
                <a
                  href="/fbo.html"
                  className="flex items-center gap-2 px-4 py-3 bg-gradient-to-r from-green-600 to-green-500 text-white rounded-lg"
                >
                  <Leaf className="w-5 h-5" />
                  <span>FBO</span>
                  <ExternalLink className="w-4 h-4 ml-auto" />
                </a>
              )}
            </div>
          </nav>
        </div>
      )}
    </header>
  );
}
