import { useEffect, useRef } from 'react';
import { FileText, ShoppingCart, Code, MapPin, ArrowRight, Download } from 'lucide-react';
import gsap from 'gsap';
import { ScrollTrigger } from 'gsap/ScrollTrigger';

gsap.registerPlugin(ScrollTrigger);

const mainServices = [
  {
    icon: FileText,
    title: 'Dématérialisation & GED',
    description: 'Capture, classement, workflow et archivage.',
  },
  {
    icon: ShoppingCart,
    title: 'E-Procurement',
    description: 'Achats, budgets, validation et reporting.',
  },
  {
    icon: Code,
    title: 'Développement sur mesure',
    description: 'Web, mobile, APIs et intégrations.',
  },
  {
    icon: MapPin,
    title: 'Géolocalisation & IoT',
    description: 'Track & trace, flotte et capteurs connectés.',
  },
];

const additionalServices = [
  {
    title: 'CONSEIL & DÉPLOIEMENT',
    description: 'Ateliers de cadrage, cartographie des processus, paramétrage, formation et conduite du changement.',
    cta: 'En savoir plus',
  },
  {
    title: 'SÉCURITÉ & CONFORMITÉ',
    description: 'Hébergement sécurisé, sauvegarde, gestion des accès, traçabilité et conformité réglementaire.',
    cta: 'En savoir plus',
  },
];

export default function Services() {
  const sectionRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const ctx = gsap.context(() => {
      gsap.fromTo(
        '.service-card',
        { y: 50, opacity: 0 },
        {
          y: 0,
          opacity: 1,
          duration: 0.6,
          stagger: 0.1,
          scrollTrigger: {
            trigger: '.services-grid',
            start: 'top 85%',
            toggleActions: 'play none none reverse',
          },
        }
      );
    }, sectionRef);

    return () => ctx.revert();
  }, []);

  return (
    <section ref={sectionRef} id="services" className="section-padding bg-[#0a0a0a] relative">
      <div className="container-custom">
        {/* Section Header */}
        <div className="mb-12">
          <span className="text-sm text-white/40 uppercase tracking-widest mb-4 block">
            Services
          </span>
          <h2 className="text-3xl md:text-4xl lg:text-5xl font-bold text-white">
            DOMAINES D'<span className="gradient-text">EXPERTISE</span>
          </h2>
        </div>

        {/* Main Services Grid */}
        <div className="services-grid grid md:grid-cols-2 lg:grid-cols-4 gap-6 mb-12">
          {mainServices.map((service, index) => (
            <div
              key={index}
              className="service-card card-hover p-6 group cursor-pointer"
            >
              <div className="w-12 h-12 rounded-lg bg-white/5 flex items-center justify-center mb-4 group-hover:bg-red-500/20 transition-colors">
                <service.icon className="w-6 h-6 text-red-500" />
              </div>
              <h3 className="text-lg font-semibold text-white mb-2">
                {service.title}
              </h3>
              <p className="text-white/50 text-sm">
                {service.description}
              </p>
            </div>
          ))}
        </div>

        {/* Download Button */}
        <div className="mb-16">
          <button className="btn-outline">
            <Download className="w-4 h-4" />
            <span>Télécharger la fiche métier</span>
          </button>
        </div>

        {/* Additional Services */}
        <div className="grid lg:grid-cols-2 gap-8">
          {additionalServices.map((service, index) => (
            <div key={index} className="service-card p-8">
              <h3 className="text-xl font-bold text-white mb-4">
                {service.title.split('&').map((part, i) => (
                  <span key={i}>
                    {i === 1 && <span className="gradient-text"> & </span>}
                    {i === 1 ? <span className="gradient-text">{part.trim()}</span> : part}
                  </span>
                ))}
              </h3>
              <p className="text-white/60 mb-6">
                {service.description}
              </p>
              <button className="text-red-500 hover:text-red-400 text-sm font-medium flex items-center gap-2 group">
                <span>{service.cta}</span>
                <ArrowRight className="w-4 h-4 group-hover:translate-x-1 transition-transform" />
              </button>
            </div>
          ))}
        </div>

        {/* Integration & Support */}
        <div className="mt-12 text-center lg:text-left">
          <h3 className="text-xl font-bold text-white mb-4">
            INTÉGRATION <span className="gradient-text">& SUPPORT</span>
          </h3>
          <p className="text-white/60 max-w-2xl mb-6">
            Connecteurs avec votre SI, automatisation, supervision et assistance : nous assurons la continuité de service au quotidien.
          </p>
          <button className="btn-outline">
            <span>Contacter le support</span>
            <ArrowRight className="w-4 h-4" />
          </button>
        </div>
      </div>
    </section>
  );
}
